#!/bin/bash

#######

src="exploit_regions.rpt"
bk="exploit_regions.rpt.back"
tmp="exploit_regions.rpt.tmp"

#######

# progress bar function
# https://unix.stackexchange.com/a/415450
prog() {
local w=1 p=$1;  shift
# create a string of spaces, then change them to dots
printf -v dots "%*s" "$(( $p * $w ))" ""; dots=${dots// /.};
# print those dots on a fixed-width space plus the percentage etc. 
printf "\r\e[K|%-*s| %3d %% %s" "$w" "$dots" "$p" "$*"; 
}

#######

echo "Initializing ..."

# get only unique regions/site coordinates
grep -E '[{].+[}]' $src | sort | uniq > $tmp

# backup and clear orig file
mv $src $bk

# keeps files content in arrays; for performance
readarray -t bk_ < $bk
readarray -t tmp_ < $tmp

# declare another, associative array; fast to access later on
declare -A metals_

## for simplicity, i.e., to avoid having to cross-check each line w/ the unique sites, just parse all the metal infos,
## including those for redundant sites; later on we use only unique sites when accessing the array
for ((i=0; i<${#bk_[@]}; i++)); do

	## Example for original rpt file syntax
	##
	#Exploit Region 1:
	#Metal 1 - 30, Metal 2 - 72, Metal 3 - 30, Metal 4 - 49, Metal 5 - 15, Metal 6 - 49
	#Sites:
	#{10.07 5.46 10.26 6.86} {10.26 5.46 10.45 6.86} {10.45 5.46 10.64 6.86} {10.64 5.46 10.83 6.86} {10.83 5.46 11.02 6.86} {11.02 5.46 11.21 6.86} {11.21 5.46 11.4 6.86} {11.4 5.46 11.59 6.86} {11.59 5.46 11.78 6.86} {11.78 5.46 11.97 6.86} {11.97 5.46 12.16 6.86} {12.16 5.46 12.35 6.86} {12.35 5.46 12.54 6.86} {12.54 5.46 12.73 6.86} {12.73 5.46 12.92 6.86} {12.92 5.46 13.11 6.86} {13.11 5.46 13.3 6.86} {13.3 5.46 13.49 6.86} {13.49 5.46 13.68 6.86} {13.68 5.46 13.87 6.86} {13.87 4.06 14.06 5.46} {13.87 5.46 14.06 6.86} {14.06 4.06 14.25 5.46} {14.06 5.46 14.25 6.86} {14.25 4.06 14.44 5.46} {14.25 5.46 14.44 6.86} {14.44 4.06 14.63 5.46} {14.44 5.46 14.63 6.86} {14.63 4.06 14.82 5.46} {14.63 5.46 14.82 6.86} {14.82 5.46 15.01 6.86} {15.01 5.46 15.2 6.86} {15.2 5.46 15.39 6.86} {15.39 5.46 15.58 6.86} {15.58 5.46 15.77 6.86} {15.77 5.46 15.96 6.86} {15.96 5.46 16.15 6.86} {16.15 5.46 16.34 6.86} {16.34 4.06 16.53 5.46} {16.34 5.46 16.53 6.86} {16.53 4.06 16.72 5.46} {16.53 5.46 16.72 6.86} {16.72 4.06 16.91 5.46} {16.72 5.46 16.91 6.86} {16.91 4.06 17.1 5.46} {16.91 5.46 17.1 6.86} {17.1 4.06 17.29 5.46} {17.1 5.46 17.29 6.86} {17.29 4.06 17.48 5.46} {17.29 5.46 17.48 6.86} {17.48 4.06 17.67 5.46} {17.48 5.46 17.67 6.86} {17.67 4.06 17.86 5.46} {4.18 4.06 4.37 5.46} {4.18 5.46 4.37 6.86} {4.18 6.86 4.37 8.26} {4.37 4.06 4.56 5.46} {4.37 5.46 4.56 6.86} {4.37 6.86 4.56 8.26} {4.56 5.46 4.75 6.86} {4.56 6.86 4.75 8.26} {4.75 5.46 4.94 6.86} {4.75 6.86 4.94 8.26} {4.94 4.06 5.13 5.46} {4.94 5.46 5.13 6.86} {4.94 6.86 5.13 8.26} {5.13 4.06 5.32 5.46} {5.13 5.46 5.32 6.86} {5.32 4.06 5.51 5.46} {5.32 5.46 5.51 6.86} {5.51 5.46 5.7 6.86} {5.7 5.46 5.89 6.86} {5.89 5.46 6.08 6.86} {6.08 5.46 6.27 6.86} {6.27 5.46 6.46 6.86} {6.46 5.46 6.65 6.86} {6.65 5.46 6.84 6.86} {6.84 5.46 7.03 6.86} {7.03 5.46 7.22 6.86} {7.22 5.46 7.41 6.86} {7.41 5.46 7.6 6.86} {7.6 5.46 7.79 6.86} {7.79 5.46 7.98 6.86} {7.98 5.46 8.17 6.86} {8.17 5.46 8.36 6.86} {8.36 5.46 8.55 6.86} {8.55 5.46 8.74 6.86} {8.74 5.46 8.93 6.86} {8.93 5.46 9.12 6.86} {9.12 5.46 9.31 6.86} {9.31 5.46 9.5 6.86} {9.5 5.46 9.69 6.86} {9.69 5.46 9.88 6.86} {9.88 5.46 10.07 6.86}
	#94 Sites

	if [[ ${bk_[$i]} != "Sites:" ]]; then
		continue
	fi

	site=${bk_[$i+1]}
	metals_[$site]=${bk_[$i-1]}
done

# obtain stats first; going over all regions
#
region_curr=0
sites_total=0
sites_max=0
tracks_curr=0
tracks_total=0
tracks_max=0

#Metal 1 - 110, Metal 2 - 13, Metal 3 - 110, Metal 4 - 9, Metal 5 - 55, Metal 6 - 9, Metal 7 - 18, Metal 8 - 3, Metal 9 - 10, Metal 10 - 2
regex_metals="((Metal )([1-9][0]*)( - )([0-9]+)(, )*)"

echo "Obtaining stats ..."

# iterate over unique sites
sites=${#tmp_[@]}
for ((i=0; i<$sites; i++)); do

	# progress bar
	prog $(( 100 * (i+1)/sites ))

	((region_curr = region_curr + 1))

	site=${tmp_[$i]}

	# site stats
	sites_curr=$(echo $site | grep -o '{' | wc -l)
	((sites_total = sites_total + sites_curr))
	sites_max=$(( sites_curr > sites_max ? sites_curr : sites_max ))

	# routing stats
	metals=${metals_[$site]}
	## eval metal layers separately, as many as there are; sum up tracks
	tracks_curr=0
	while [[ "$metals" =~ $regex_metals ]]; do
#		declare -p BASH_REMATCH

		# update string, by cutting off match that is in front
		metals=$(echo $metals | sed -n -e "s/${BASH_REMATCH[0]}//p")
#		echo $metals

		tracks_curr=$(bc <<< "$tracks_curr + ${BASH_REMATCH[5]}")
#		echo $tracks_curr
	done

	((tracks_total = tracks_total + tracks_curr))
	tracks_max=$(( tracks_curr > tracks_max ? tracks_curr : tracks_max ))

done

# final newline for progressbar
echo "" 

# sanity check for no regions at all
if [[ $region_curr != 0 ]]; then
	sites_avg=$(bc <<< "scale=2; ($sites_total / $region_curr)")
	tracks_avg=$(bc <<< "scale=2; ($tracks_total / $region_curr)")
else
	sites_avg=0
	tracks_avg=0
fi

echo "Write back rpt file ..."

echo "Regions: $region_curr" >> $src
echo "Max free tracks across regions: $tracks_max" >> $src
echo "Avg free tracks across regions: $tracks_avg" >> $src
echo "Total free tracks across regions: $tracks_total" >> $src
echo "Max sites across regions: $sites_max" >> $src
echo "Avg sites across regions: $sites_avg" >> $src
echo "Total sites across regions: $sites_total" >> $src

# print regions
#
region_curr=0
sites=${#tmp_[@]}

for ((i=0; i<$sites; i++)); do

	# progress bar
	prog $(( 100 * (i+1)/sites ))

	site=${tmp_[$i]}
	sites_curr=$(echo $site | grep -o '{' | wc -l)

	((region_curr = region_curr + 1))

	echo "" >> $src
	echo "Region: $region_curr" >> $src
	echo "Free tracks: ${metals_[$site]}" >> $src
	echo "Sites count: $sites_curr" >> $src
	echo "Sites coordinates: $site" >> $src

done

# final newline for progressbar
echo "" 

rm $tmp
